const path = require('path');

// Load environment variables
require('dotenv').config();

const config = {
  // Server configuration
  server: {
    port: process.env.PORT || 3001,
    host: process.env.HOST || 'localhost',
    nodeEnv: process.env.NODE_ENV || 'development'
  },

  // Database configuration
  database: {
    path: process.env.DB_PATH || path.join(__dirname, '..', 'database', 'chat.db'),
    connectionTimeout: parseInt(process.env.DB_TIMEOUT) || 30000,
    maxConnections: parseInt(process.env.DB_MAX_CONNECTIONS) || 10
  },

  // Session configuration
  session: {
    secret: process.env.SESSION_SECRET || 'fallback-secret-key-change-in-production',
    maxAge: parseInt(process.env.SESSION_MAX_AGE) || 24 * 60 * 60 * 1000, // 24 hours
    secure: process.env.NODE_ENV === 'production'
  },

  // Google OAuth configuration
  google: {
    clientId: process.env.GOOGLE_CLIENT_ID,
    clientSecret: process.env.GOOGLE_CLIENT_SECRET,
    callbackURL: process.env.GOOGLE_CALLBACK_URL || '/auth/google/callback'
  },

  // File upload configuration
  upload: {
    maxFileSize: parseInt(process.env.MAX_FILE_SIZE) || 10 * 1024 * 1024, // 10MB
    allowedMimeTypes: [
      'image/jpeg',
      'image/png', 
      'image/gif',
      'image/webp',
      'application/pdf',
      'text/plain',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    ],
    uploadDir: path.join(__dirname, '..', 'uploads')
  },

  // Rate limiting configuration
  rateLimit: {
    api: {
      windowMs: 60 * 1000, // 1 minute
      maxRequests: 100
    },
    auth: {
      windowMs: 60 * 1000, // 1 minute
      maxRequests: 20
    },
    socket: {
      message: {
        windowMs: 60 * 1000, // 1 minute
        maxRequests: 30
      },
      privateMessage: {
        windowMs: 60 * 1000, // 1 minute
        maxRequests: 20
      },
      typing: {
        windowMs: 60 * 1000, // 1 minute
        maxRequests: 60
      }
    }
  },

  // Logging configuration
  logging: {
    level: process.env.LOG_LEVEL || 'info',
    logDir: path.join(__dirname, '..', 'logs'),
    maxFileSize: '10MB',
    maxFiles: 5
  },

  // Security configuration
  security: {
    corsOrigin: process.env.CORS_ORIGIN || '*',
    trustProxy: process.env.TRUST_PROXY === 'true',
    helmet: {
      contentSecurityPolicy: process.env.NODE_ENV === 'production',
      crossOriginEmbedderPolicy: false
    }
  },

  // Message configuration
  message: {
    maxLength: parseInt(process.env.MAX_MESSAGE_LENGTH) || 5000,
    minLength: 1,
    allowedTypes: ['text', 'file', 'image']
  },

  // User configuration
  user: {
    nameMinLength: 2,
    nameMaxLength: 100,
    maxOnlineUsers: parseInt(process.env.MAX_ONLINE_USERS) || 1000
  },

  // Development configuration
  development: {
    enableDebugLogs: process.env.DEBUG_LOGS === 'true',
    mockAuth: process.env.MOCK_AUTH === 'true',
    seedDatabase: process.env.SEED_DB === 'true'
  }
};

// Validation
if (!config.google.clientId || !config.google.clientSecret) {
  if (config.server.nodeEnv === 'production') {
    throw new Error('Google OAuth credentials are required in production');
  } else {
    console.warn('Warning: Google OAuth credentials not configured');
  }
}

if (config.session.secret === 'fallback-secret-key-change-in-production' && config.server.nodeEnv === 'production') {
  throw new Error('SESSION_SECRET must be set in production');
}

module.exports = config;